<?php
/**
 * Database communication wrapper
 * 
 * @author Ivan Dlugoš
 * @copyright Ivan Dlugoš
 *
 * @todo - debugging - chyby, posledne query, atd
 */
class Db {
	private static $instance;
  const QUERY_SELECT = 'select';
  const QUERY_INSERT = 'insert';
  const QUERY_UPDATE = 'update';
  const QUERY_COMMAND = 'command';
  const QUERY_DELETE = self::QUERY_UPDATE;
  const QUERY_UNDEFINED = self::QUERY_SELECT;
  
  /**
   * Configuration
   * tbprefix - prefix of each table in db
   * cache_copies - copies of files per db result
   * cache_db_ration - ratio between using cache or db - 10 means no cache 
   */
  public $configuration = array('tbprefix' => '');
  public $conn;
  public $errors;
  
  private $use_cache = false;
  private $logs = array('errors' => array(), 'queries' => array());
  private $checksums = array();
  private $allowed_select_additional = array('calc_found_rows' => 'SQL_CALC_FOUND_ROWS');
  private $multi_query = array();
  private $multi_query_prepare = false;
  private $last_error = null;
  public $last_query = null;
  
  private function __construct(){
    $this->prepare_errors();
    if(!$this->mysql_connect(DB_HOST, DB_LOGIN, DB_PASSWORD, DB_DBNAME, DB_PDO_DSN)){return FALSE;}      // establish connection to MySQL, if it could not be established, return FALSE;
  }
  
	private function __clone(){}
  
	public static function &get_instance(){
		if(self::$instance === null){
			self::$instance = new db();
		}
		return self::$instance;
	}
	
  public function __destruct(){
    unset($this->conn);
    if($this->save_logs($this->logs) === false){
      trigger_error("NEPODARILO SA ZÁLOHOVAŤ DB LOG", E_USER_WARNING);
    }
    if(DEBUG){
      $empty = true;
      foreach($this->errors as $errs){
        if(!empty($errs)) $empty = false;
      }
      if(!$empty) vd($this->errors);
    }
  }
       
  /**
   * Makes database connection
   * 
   * @param string $host
   * @param string $username
   * @param string $passwd
   * @param string $dbname
   * 
   * @return boolean
   */
  private function mysql_connect($host, $username, $password, $dbname, $dsn){
    try {
      $this->conn = new PDO($dsn, $username, $password);
    } catch (PDOException $e) {
      $this->last_error = $e;
      $this->errors['sql'] = 'Connection can not be established!';
      $err = "Pri behu metódy " . __METHOD__ . " triedy " . __CLASS__ . " sa nepodarilo pripojiť k databáze. PDO vrátil chybu (" . $e->getCode() . "): {" . $e->getMessage() . "}.";
      trigger_error($err, E_USER_WARNING);
      return false;
    }
    $this->conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $this->conn->setAttribute(PDO::ATTR_AUTOCOMMIT, true);
    $this->query("SET names 'utf8';", self::QUERY_COMMAND);  // SET character_set_client=utf8; SET character_set_connection=utf8; SET character_set_results=utf8
    return true;
  }
  
  public function getLastErrorCode(){
  	return !empty($this->last_error) ? $this->last_error->getCode() : 0;
  }
  
  private function query($query, $type = self::QUERY_UNDEFINED){
    $this->last_query = $query;
    if(!method_exists($this, 'query_'.$type)){
    	trigger_error('Undefined query type called (' . $type . ')');
      $type = self::QUERY_UNDEFINED;
    }
    
    return call_user_func_array(array($this, 'query_'.$type), array($query));
  }
  
  private function query_select($query){
    try {
      $result = $this->conn->query($query);
      $return = array();
      foreach($result->fetchAll(PDO::FETCH_ASSOC) as $row){
        $return[] = $row;
      }
      $this->log_query($query, 'Rows in result set: ' . count($return));
      return $return;
      
    } catch (PDOException $e){
      $this->log_error($query, $e->getCode(), $e->getMessage(), $e);
      $this->errors['sql'][] = $this->last_error = $e;
      return false;
    }
  }
  
  private function query_insert($query){
    try{
      $res = $this->conn->exec($query);
      $this->log_query($query, 'Last insert ID: ' . $this->conn->lastInsertId());
      return $this->conn->lastInsertId();
    } catch (PDOException $e){
      $this->log_error($query, $e->getCode(), $e->getMessage(), $e);
      $this->errors['sql'][] = $this->last_error = $e;
      return false;
    }
  }
  
  private function query_update($query){
    try{
      $res = $this->conn->exec($query);
      $this->log_query($query, 'Affected rows: ' . $res);
      return $res;
    } catch (PDOException $e){
      $this->log_error($query, $e->getCode(), $e->getMessage(), $e);
      $this->errors['sql'][] = $this->last_error = $e;
      return false;
    }
  }
  
  private function query_command($query){
    try {
      $result = $this->conn->query($query);
      $this->log_query($query, 'No result');
      return $result;
    } catch (PDOException $e){
      $this->log_error($query, $result->errorCode(), $result->errorInfo(), $e);
      $this->errors['sql'][] = $this->last_error = $e;
      return false;
    }
  }
    
  /**
   * Loads whole table (or columns) from db
   * @param array $conf - configuration
   * @param array $tables - array('alias'=>'table')
   * @param array $columns - array(array('columnname', 'tablename', 'columnalias'));
   * @param array $where - Where conditions - array(group => array('goperator' => (AND|OR|XOR), conditions => array('left' => string, 'right' => string, 'operator' => string(=|like|>...))))
   * @param array $limit - Limit configuration -  array('count' => int, 'offset' => int)
   * @param array $order - Order by configuration - array('column' => string, 'dir' => string(asc|desc))
   * @param array $group - Group by configuration - array(array('column' => null, 'dir' => 'asc'), 'rollup' => false)
   * @return array result_set or false on failure
   */
  public function select($conf, $tables, $columns, $where = null, $limit = null, $order = null, $group = null){
  	if($limit === null) $limit = array('count' => null, 'offset' => 0);
    $this->check_conf($conf, 'select');
    
    if ((!isset($limit['count']) AND ($limit['count'] !== null)) AND (intval($limit['count']) != $limit['count'])){
      $this->errors['standard'][] = 'Limit not set or wrong "count" parameter given.';
      if (!isset($limit['offset']) OR !is_numeric($limit['offset'])){
        $limit['offset'] = 0;
        $this->errors['optional'][] = 'Limit offset not set or given value is not an integer.';
      }
    }
    if (($conf['distinct'] != 'ALL') AND ($conf['distinct'] != 'DISTINCT') AND ($conf['distinct'] != 'DISTINCTROW')){
      $conf['distinct'] = 'ALL';
      $this->errors['standard'][] = 'Wrong distinct type given.';
    }
    if (($conf['priority'] != '') AND ($conf['priority'] != 'HIGH_PRIORITY')){
      $conf['priority'] = '';
      $this->errors['standard'][] = 'Wrong priority type given.';
    }
    if (($conf['result'] != '') AND ($conf['result'] != 'SQL_SMALL_RESULT') AND ($conf['result'] != 'SQL_BIG_RESULT') AND ($conf['result'] != 'SQL_BUFFER_RESULT')){
      $conf['result'] = '';
      $this->errors['standard'][] = 'Wrong result type given.';
    }
    if (($conf['cache'] != '') AND ($conf['result'] != 'SQL_CACHE') AND ($conf['result'] != 'SQL_NO_CACHE')){
      $conf['cache'] = '';
      $this->errors['standard'][] = 'Wrong cache type given.';
    }
    
    $additional = '';
    foreach($conf as $k => $val){
      if(!is_numeric($k)) continue;
      if(!empty($this->allowed_select_additional[$val])){
        $additional .= $this->allowed_select_additional[$val] . ' ';
      }
    }
    
    if($group !== null){
      if(!is_array($group)){
        $this->errors['standard'][] = 'Group by not set - wrong type given.';
        $groupby = '';
      } else {
        if (!isset($group['rollup']) OR !is_bool($group['rollup'])){
          $group['rollup'] = '';
        } elseif($group['rollup']) {
          $group['rollup'] = 'WITH_ROLLUP';
        } else {
          $group['rollup'] = '';
        }
        
        $groupby = 'GROUP BY ';
        foreach($group as $exprk => $expr){
          if ($exprk === 'rollup'){
            continue;
          }
          
          $groupby .= "`{$expr}`,";
        }
        
        if($groupby == 'GROUP BY '){
          $groupby = '';
        } elseif($group['rollup'] != '') {
          $groupby = trim($groupby, ',') . ' ' . $group['rollup'];
        } else {
        	$groupby = trim($groupby, ',');
        }
      }
    } else {
      $groupby = '';
    }
    
    if($order !== null){
      if(!is_array($order)){
        $this->errors['standard'][] = 'Order by not set - wrong type given.';
        $orderby = '';
      } else {
        $orderby = 'ORDER BY ';
        foreach($order as $expr){
          if(empty($expr[0])){
            $this->errors['standard'][] = 'Order by not set or wrong column parameter given. Subexpression skipped';
            continue;
          }
          if (!isset($expr['dir']) OR ((($expr['dir'] != 'asc') AND ($expr['dir'] != 'desc')))){
            $expr['dir'] = 'asc';
            $this->errors['optional'][] = 'Order by direction not set or not from (asc|desc).';
          }
          $expr['dir'] = strtoupper($expr['dir']);
          $orderby .= "`{$expr[0]}` {$expr['dir']},";
        }
        if($orderby == 'ORDER BY '){
          $orderby = '';
        } else {
          $orderby = trim($orderby, ',');  
        }
      }
    } else {
      $orderby = '';
    }
    
    if (!is_array($tables)){
      $this->errors['fatal'][] = 'Tables not set - wrong parameter format given.';
      return FALSE;
    } else {
      $tout = '';
      foreach($tables as $alias => $table){
        $alias = is_numeric($alias) ? '' : " AS `{$alias}`";
        $tout .= "`{$this->configuration['tbprefix']}{$table}`{$alias},";    
      }
      $tout = rtrim($tout, ',');
    }
    if(count($tables) == 1){
      reset($tables);  
      $tbl = current($tables);
    }
    
    if (!is_array($columns)){
      if($columns == '*'){
        $cout = '*';
      } else {
        $this->errors['fatal'][] = 'Columns not set - wrong parameter format given.';
        return FALSE;
      }
    } else {
      $cout = '';
      foreach($columns as $column){
        if (!is_array($column) OR !isset($column[0]) OR (!isset($column[1]) AND !isset($tbl))){
          $this->errors['fatal'][] = 'Wrong column parameter format given.';
          return FALSE;
        }
        $column[2] = isset($column[2]) ? " AS `{$column[2]}`" :  '';
        if(isset($tbl) AND empty($column[1])){
          $column[1] = $tbl;
        }
        $key = array_search($column[1], $tables);
        if(($key !== FALSE) AND !is_numeric($key)){
          $column[1] = $key;  
        }
        if($column[0] != '*'){
          if($conf['transform_dt'] AND (substr($column[0],0,2) == 'dt')){
            if(empty($column[2])){
              $column[2] = " AS `{$column[0]}`";
            }
            $col = "DATE_FORMAT(`{$this->configuration['tbprefix']}{$column[1]}`.`{$column[0]}`, '%d. %m. %Y %H:%i:%s')";
          } else {
            $col = "`{$this->configuration['tbprefix']}{$column[1]}`.`{$column[0]}`";
          }
        }
        $cout .= "{$col}{$column[2]},";
      }
      $cout = rtrim($cout, ',');
    }    
    
    $conf['distinct'] = strtoupper($conf['distinct']);
    $conf['priority'] = strtoupper($conf['priority']);
    $conf['result'] = strtoupper($conf['result']);
    $conf['cache'] = strtoupper($conf['cache']);
    
    if(isset($where['join'])){
      $join = $where['join'];
      unset($where['join']);
      $join_tables = ''; 
      foreach($join['tables'] as $key => $value){
        if(is_numeric($key)){
          $join_tables .= "`$value`,";
        } else {
          $join_tables .= "`$value` as `$key`,";
        }
      }
      $join = strtoupper($join['type']) . ' JOIN ('. rtrim($join_tables,',') .') ON ' . $this->condition_parser($join['on'], $this->errors);
    } else {
      $join = '';
    }
    if(isset($where['goperator']) AND (count($where) == 1)){
      $where = '';
    } else {
      $where = $this->condition_parser($where, $this->errors);
    }
    
    if(!empty($where)){
      $where = 'WHERE ' . $where;
    }
    
    if($limit['count'] == null){
      $limit = '';
      $limit_wanted = null;
    } else {
      $limit_wanted = $limit['count'];
      $limit = "LIMIT {$limit['offset']},{$limit['count']}";
    }

    $query = "SELECT {$additional} {$conf['distinct']} {$conf['priority']} {$conf['result']} {$conf['cache']} {$cout} FROM {$tout} {$join} {$where} {$groupby} {$orderby} {$limit}";

    if(true === $this->multi_query_prepare){
      if(!empty($conf['mq_id']) AND !isset($this->multi_query[$conf['mq_id']])){
        $this->multi_query[$conf['mq_id']] = array('query' => $query, 'mq_id' => $conf['mq_id'], 'type' => 'select', 'conf' => $conf);
      } else {
        $this->multi_query[] = array('query' => $query, 'type' => 'select', 'conf' => $conf);
      }
      return;  
    }
    
    $result = $this->query($query, self::QUERY_SELECT);
    
    if($result !== false){
      if(!empty($conf['id'])){
        $this->assign_id($result, $conf['id']);
      }
    }
    
    if(($limit_wanted == 1) AND is_array($result) AND count($result == 1)){
      $result = current($result);
    }
    
    return $result;
  }
    
  /**
   * Inserts record to db
   * @param array $conf - array('priority' => '','ignore' => '')
   * @param string $table
   * @param array $values - array('columns' => array('column1', 'column2',...) 'values' => array(array('1value1', '1value2', '{DEFAULT}'), array('2value1', '2value2', '2value3')))
   * @param array $on_duplicate - array('column' => 'value', 'column2 => 'value2',...)
   * @return integer last_insert_id, boolean false on failure 
   */
  public function insert($conf, $table, $values, $on_duplicate = null){
    $this->check_conf($conf, 'insert');
    if (($conf['priority'] != '') AND ($conf['priority'] != 'HIGH_PRIORITY') AND ($conf['priority'] != 'LOW_PRIORITY') AND ($conf['priority'] != 'DELAYED')){
      $conf['priority'] = '';
      $this->errors['standard'][] = 'Wrong priority type given.';
    }
    if (($conf['ignore'] != '') AND ($conf['ignore'] != 'IGNORE')){
      $conf['ignore'] = '';
      $this->errors['standard'][] = 'Ignore parameter set do wrong value.';
    }
    if (empty($table)){
      $this->errors['fatal'][] = 'Table not set.';
      return FALSE;
    } 
    if (empty($values) or !is_array($values)){
      $this->errors['fatal'][] = 'Insert set not set or not an array.';
      return FALSE;
    }
    if (empty($values['columns']) or !is_array($values['columns'])){
      //$this->errors['fatal'][] = 'Columns not set or not an array.';
      //return FALSE;
      $values['columns'] = array_keys($values);
    }
    if (empty($values['values']) or !is_array($values['values'])){
      //$this->errors['fatal'][] = 'Values not set or not an array.';
      //return FALSE;
      $values['values'] = array(array_diff_key($values, array('columns' => null)));
      foreach($values as $k => $v){
        if(in_array($k, $values['columns'])) unset($values[$k]);
      }
    }
    if (($on_duplicate !== null) AND (!is_array($on_duplicate))){
      $this->errors['fatal'][] = 'On duplicate key not set or not an array.';
      return FALSE;
    }
    
    $columns = '';
    $num_of_columns = 0;
    $expressions = '';
    $count_of_inserts = 0;

    foreach($values as $key => $value){
      if($key == 'columns'){
        foreach($value as $col){
          $num_of_columns++;
          $columns .= "`{$col}`,";
        }
      } elseif($key == 'values'){
        $count_of_inserts = count($value);
        foreach($value as $values_set){
          $expressions .= '(';
          if (count($values_set) != $num_of_columns){
            $this->errors['fatal'][] = 'Count of columns not equal to count of values.';
            return FALSE;
          }
          foreach($values_set as $expr){
            if (is_string($expr) AND (strlen($expr) > 0) AND ($expr[0] == '{') AND ($expr[strlen($expr)-1] == '}')){
              $expressions .= substr($expr, 1, -1) . ',';
            } else {
              $expressions .= $this->escape_string($expr) . ",";
            }
          }
          $expressions = rtrim($expressions, ',') . '),';
        }
      } else {
        $this->errors['fatal'][] = 'Unexpected identifier in values_set.';
        return FALSE;
      }
    }
    
    $columns = rtrim($columns, ',');
    $expressions = rtrim($expressions, ',');
    
    if (($on_duplicate !== null) AND ($count_of_inserts != 1)){
      $this->errors['fatal'][] = 'On duplicate key can not be used with more then one row insertion.';
      return FALSE;
    }
    
    if($on_duplicate !== null){
      $on_dupl = 'ON DUPLICATE KEY UPDATE ';
      foreach($on_duplicate as $col => $expr){
        if (($expr[0] == '{') AND ($expr[strlen($expr)-1] == '}')){
          $expressions .= "`{$col}`=" . substr($expr, 1, -1) . ',';
        } else {
          $on_dupl .=  "`{$col}`=" . $this->escape_string($expr) . ",";
        }
      }
      $on_dupl = rtrim($on_dupl, ',');
    } else {
      $on_dupl = '';    
    }
    
    $conf['priority'] = strtoupper($conf['priority']);
    $conf['ignore'] = strtoupper($conf['ignore']);
    
    $query = "INSERT {$conf['priority']} {$conf['ignore']} INTO `{$table}` ({$columns}) VALUES {$expressions} {$on_dupl}";

    $result = $this->query($query, self::QUERY_INSERT);
    if($result === false){
      return false;
    } elseif ($on_duplicate === null){
      return $result;
    } else {
      return;
    }
  }
    
  /**
   * Updates single row
   * @param array $conf - array('priority' => '','ignore' => '')
   * @param string $table
   * @param array $values - array('column' => 'value', 'column2' => 'value2',...)
   * @param array $where - where conditions - array(group => array('goperator' => (AND|OR|XOR), conditions => array('left' => string, 'right' => string, 'operator' => string(=|like|>...))))
   * @param integer $limit
   * @param array $order - Order by configuration - array('column' => string, 'dir' => string(asc|desc))
   */
  public function update($conf, $table, $values, $where = null, $limit = null, $order = array('column' => null, 'dir' => 'asc')){
    $this->check_conf($conf, 'update');
    if (($conf['priority'] != '') AND ($conf['priority'] != 'LOW_PRIORITY')){
      $conf['priority'] = '';
      $this->errors['standard'][] = 'Wrong priority type given.';
    }
    if (($conf['ignore'] != '') AND ($conf['ignore'] != 'IGNORE')){
      $conf['ignore'] = '';
      $this->errors['standard'][] = 'Ignore parameter set do wrong value.';
    }
    if (empty($order['column']) AND ($order['column'] !== null)){
      $this->errors['standard'][] = 'Ordering not set or wrong column parameter given.';
      if (!isset($order['dir']) OR ((($order['dir'] != 'asc') AND ($order['dir'] != 'desc')))){
        $order['dir'] = 'asc';
        $this->errors['optional'][] = 'Ordering direction not set or not from (asc|desc).';
      }
    }
    if (($limit !== null) AND (intval($limit) != $limit)){
      $this->errors['standard'][] = 'Limit not set or not a number.';
    }
    if (empty($table)){
      $this->errors['fatal'][] = 'Table not set.';
      return false;
    }
    if (empty($values) OR !is_array($values)){
      $this->errors['fatal'][] = 'Update set not set or not an array.';
      return false;
    }
    
    $expressions = '';
    foreach($values as $column => $value){
    	if(strlen($value) == 0){
    		$expressions .= "`{$column}`='',";
    	} elseif (($value[0] == '{') AND ($value[strlen($value)-1] == '}')){
        $expressions .= "`{$column}`=" . substr($value, 1, -1) . ",";
      } else {
        $expressions .=  "`{$column}`=" . $this->escape_string($value) . ",";
      }
    }
    $expressions = rtrim($expressions, ',');
    
    $conf['priority'] = strtoupper($conf['priority']);
    $conf['ignore'] = strtoupper($conf['ignore']);
    $where = $this->condition_parser($where, $this->errors);
    if(!empty($where)){
      $where = "WHERE " . $where;
    }
    $order = ($order['column'] == null) ? '' : "ORDER BY `{$order['column']}` {$order['dir']}";
    $limit = ($limit['count'] == null) ? '' : "LIMIT " . intval($limit);
    
    $query = "UPDATE {$conf['priority']} {$conf['ignore']} `{$table}` SET {$expressions} {$where} {$order} {$limit}";

    return $this->query($query, self::QUERY_UPDATE);
  }
  
  /**
   * Updates multiple tables
   * @param array $conf - array('priority' => '','ignore' => '')
   * @param array $tables_values - array('table_name' => array('column' => 'value', 'column2' => 'value2',..) 'table2' = array(...))
   * @param array $where - Where conditions - array(group => array('goperator' => (AND|OR|XOR), conditions => array('left' => string, 'right' => string, 'operator' => string(=|like|>...))))
   * @return integer affected_rows or boolean FALSE on failure
   */
  public function update_multi($conf, $tables_values, $where = null){
    $this->check_conf($conf, 'update');
    if (($conf['priority'] != '') AND ($conf['priority'] != 'LOW_PRIORITY')){
      $conf['priority'] = '';
      $this->errors['standard'][] = 'Wrong priority type given.';
    }
    if (($conf['ignore'] != '') AND ($conf['ignore'] != 'IGNORE')){
      $conf['ignore'] = '';
      $this->errors['standard'][] = 'Ignore parameter set do wrong value.';
    }
    if (empty($tables_values) or !is_array($tables_values)){
      $this->errors['fatal'][] = 'Update set not set or not an array.';
      return FALSE;
    }
    
    $tables = '';
    $expressions = '';
    
    foreach($tables_values as $table => $columns){
      if (is_numeric($table)) {
        $this->errors['fatal'][] = 'Bad table name given (numeric key).';
        return FALSE;
      }
      $tables .= "`{$table}`,";
      foreach($columns as $column => $value){
        if ($value == '{DEFAULT}'){
          $expressions .= "`{$table}`.`{$column}`=DEFAULT,";
        } elseif (strpos($value, '`') === false){
          $expressions .=  "`{$table}`.`{$column}`='{$value}',";
        } else {
          $expressions .=  "`{$table}`.`{$column}`={$value},";
        }
      }
    }
    
    $tables = rtrim($tables, ',');
    $expressions = rtrim($expressions, ',');
    $conf['priority'] = strtoupper($conf['priority']);
    $conf['ignore'] = strtoupper($conf['ignore']);
    $where = $this->condition_parser($where, $this->errors);
    if(!empty($where)){
      $where = "WHERE " . $where;
    }
    $query = "UPDATE {$conf['priority']} {$conf['ignore']} {$tables} SET {$expressions} {$where}";
    
    return $this->query($query, self::QUERY_UPDATE);
  }
  
  public function delete($conf, $table, $where = null, $limit = null, $order = array('column' => null, 'dir' => 'asc')){
    $this->check_conf($conf, 'delete');
    if (($conf['priority'] != '') AND ($conf['priority'] != 'LOW_PRIORITY')){
      $conf['priority'] = '';
      $this->errors['standard'][] = 'Wrong priority type given.';
    }
    if (($conf['ignore'] != '') AND ($conf['ignore'] != 'IGNORE')){
      $conf['ignore'] = '';
      $this->errors['standard'][] = 'Ignore parameter set do wrong value.';
    }
    if (empty($order['column']) AND ($order['column'] !== null)){
      $this->errors['standard'][] = 'Ordering not set or wrong column parameter given.';
      if (!isset($order['dir']) OR ((($order['dir'] != 'asc') AND ($order['dir'] != 'desc')))){
        $order['dir'] = 'asc';
        $this->errors['optional'][] = 'Ordering direction not set or not from (asc|desc).';
      }
    }
    if (($limit !== null) AND (intval($limit) != $limit)){
      $this->errors['standard'][] = 'Limit not set or not a number.';
    }
    if (empty($table)) {
      $this->errors['fatal'][] = 'Table not set.';
      return false;
    }
    
    $conf['priority'] = strtoupper($conf['priority']);
    $conf['ignore'] = strtoupper($conf['ignore']);
    $where = $this->condition_parser($where, $this->errors);
    if(!empty($where)){
      $where = "WHERE " . $where;
    }
    $order = ($order['column'] == null) ? '' : "ORDER BY `{$order['column']}` {$order['dir']}";
    $limit = ($limit['count'] == null) ? '' : "LIMIT " . intval($limit);
    
    $query = "DELETE {$conf['priority']} {$conf['ignore']} FROM `{$table}` {$where} {$order} {$limit}";

    return $this->query($query, self::QUERY_DELETE);
  }
  
  public function sdelete($conf, $table, $where = null, $limit = null, $order  = array('column' => null, 'dir' => 'asc')){
    return $this->update($conf, $table, array('status' => '0', 'id_user_deleted' => $this->_user->id, 'dt_deleted' => '{NOW()}'), $where, $limit, $order);    
  }
  
  /**
   * Parses given conditions and combines given conditions
   * 
   * @param array $condition - array('goperator' => (AND|OR|XOR), conditions => array('left' => string, 'right' => string, 'operator' => string(=|like|>...)))
   * @param array $this->errors
   * @return string condition
   */
  private function condition_parser($condition){
    if ($condition === null){
      return;
    }
    if (!isset($condition['goperator'])){      // bottom point of recursion
    	if (count($condition) == 1){
    		$subcond = reset($condition);
    		if(is_array($subcond)){
    			return $this->condition_parser($subcond);
    		} else {
    			return $subcond;
    		}
    	} elseif (count($condition) != 3){
        $this->errors['fatal'][] = "Condition parser - wrong parameter count - needed 3, " . count($condition) . " given. FALSE";
        return '(1=2)';
      }
      return "({$condition[0]} {$condition[1]} {$condition[2]})";  
    } elseif(in_array($condition['goperator'], array('OR', 'AND', 'XOR', 'or', 'and', 'xor'))) {
      $conds = array();
      $operator = strtoupper($condition['goperator']);
      unset($condition['goperator']);
      foreach ($condition as $group){
        $conds[] = $this->condition_parser($group);
      }
      return '(' . implode($operator, $conds) . ')';
    } else {
      $this->errors['fatal'][] = 'Condition parser: Missing group operator. FALSE';
      return '(1=2)';                // FALSE
    }
  }
    
  /**
   * Prepares local error var
   * @return void
   */
  private function prepare_errors(){
    if (!isset($this->errors)){
      $this->errors = array();
    }
    if (!isset($this->errors['standard'])){
      $this->errors['standard'] = array();
    }
    if (!isset($this->errors['optional'])){
      $this->errors['optional'] = array();
    }
    if (!isset($this->errors['fatal'])){
      $this->errors['fatal'] = array();
    }
    if (!isset($this->errors['sql'])){
      $this->errors['sql'] = array();
    }
  }
  
  /**
   * Check if passed configuration contains all needed values
   * @param array $conf
   * @return void
   */
  private function check_conf(&$conf, $type){
    switch ($type){
      case 'select': 
        $needed = array(
          'distinct' => 'ALL',
          'result' => '',
          'cache' => '',
          'priority' => '',
          'transform_dt' => false
          );
        break;
      case 'insert':
        $needed = array(
          'priority' => '',
          'ignore' => ''
          );
        break;
      case 'update':
        $needed = array(
          'priority' => '',
          'ignore' => ''
          );
        break;
      case 'delete':
        $needed = array(
          'priority' => '',
          'ignore' => ''
        );
        break;
    }
    foreach ($needed as $key => $value){
      if (!isset($conf[$key])){
        $conf[$key] = $value;
      }
    }
  }

  /**
   * Logs queries to internal log
   * @return void
   */
  private function log_query($query, $result){
    $this->logs['queries'][] = array('query' => trim($query), 'result' => $result, 'dt' => time());
  }
  
  /**
   * Logs errors to internal log
   * @return void
   */
  private function log_error($query, $errno, $error, $e = null){
    if(DEBUG) vd(array('errno' => $errno, 'error' => $error, 'query' => trim($query), 'exception' => $e, 'dt' => time()));
    
    $this->logs['errors'][] = array('errno' => $errno, 'error' => $error, 'query' => trim($query), 'exception' => $e, 'dt' => time());
  }
    
  private function save_logs(&$logs){
    $successful = true;
    foreach($logs as $type => &$log){
      $data = call_user_func_array(array($this, 'prepare_log_' . $type), array($log));
      if(empty($data)){
        continue;
      }
      $type = strtoupper($type);
      $path = eval('return DIRECTORY_LOGS_DB_' . $type . '."/".str_replace("{date}", date(LOG_DATE_FORMAT), DB_' . $type . '_LOG_FILENAME);');
      $successful = ($this->save_log_data($data, $path) AND $successful);
    }
    return $successful;
  }
  
  private function prepare_log_queries(&$log){
    if(!DB_LOG_SUCCESSFUL) return;
    $data = '';
    foreach($log as $entry){
      $entry['query'] = str_replace(array("\r", "\n"), array('\r', '\n'), $entry['query']);
      $data .= "\"{$entry['query']}\"\r\n  {$entry['result']}\r\n\r\n";
    }
    return $data;
  }
  
  private function prepare_log_errors(&$log){
    $data = '';
    foreach($log as $entry){
      $entry['query'] = str_replace(array("\r", "\n"), array('\r', '\n'), $entry['query']);
      if($entry['exception'] != null){
        $additional = (string) $entry['exception'] . "\r\n";
      } else {
        $additional = '';
      }
      
      $data .= "{$entry['errno']}::{$entry['error']}\r\nQuery:\"{$entry['query']}\"\r\n$additional\r\n";
    }
    return $data;
  }
  
  private function save_log_data($data, $path){
    $r = file_append($path, "============================== " . dtnow('error') . " ==============================\r\n" . $data . "===================================================================================\r\n");
    return ($r !== false);    
  }
  
  /**
   * Makes database result an associative array with id as a key
   * @param array $result_set
   * @param string $id
   */
  private function assign_id(&$result_set, $id){
    $out = array();
    $ok = false;
    foreach($result_set as $row){
      if(!$ok AND !array_key_exists($id, $row)){
      $this->errors['standard'][] = 'Passed key does not exists in result set.';
      unset($out);
      return false;
    } else {
      $ok = true;
    }
      $out[$row[$id]] = $row;
    }
    $result_set = $out;
  }

  public function sanitize_input($value){
  	if(strlen($value) == 0) return '';
    if (($value[0] == '{') AND ($value[strlen($value)-1] == '}')){
      return substr($value, 1, -1);
    } else {
      return $value;
    }
  }
  
  public function escape_string($value){
    return $this->conn->quote($value);
  }

  /**
   * Prepares set for use in sql - preserving limitation of max 64 members (MySQL)
   */
  public function prepare_set_stmt($needles, $column, $positive = true, $div = 64){
  	if(empty($needles)) return array('goperator' => 'or', array(0, $positive ? '=' : '!=', 1));
  	
    sort($needles);
    
    if(count($needles) == 1){
      return array('goperator' => 'or', array($column, $positive ? '=' : '!=', "'" . current($needles) . "'"));
    }
    
    $out = array('goperator' => 'or');
    $i = 1;
    $tmp = '(';
    $all = count($needles);
    
    foreach($needles as $value){
      $tmp .= $this->escape_string($value) . ",";
      if($i >= $div){
        $i = 0;
        $all -= $div;
        $out[] = array($column, $positive ? 'IN' : 'NOT IN', substr($tmp,0,-1).')');
        $tmp = '(';
      } elseif($i >= $all){
        $out[] = array($column, $positive ? 'IN' : 'NOT IN', substr($tmp,0,-1).')');
      }
      $i++;
    }
    
    return $out;
  }

  // TODO test
  public function calc_found_rows(){
    return $this->query('SELECT FOUND_ROWS()', self::QUERY_SELECT);
  }

  // TODO change - PDO
  public function multi_query_init(){
    if($this->multi_query_prepare) return false;
    $this->multi_query_prepare = true;
    $this->multi_query = array();
    return true;
  }
  
  public function multi_query_do(){
    $query = '';
    foreach($this->multi_query as $key => $info){
      $query .= $info['query'] . ';';
    }
    if(empty($query)){
      $return = FALSE;
    }
    
    if(false === $this->mysqli->multi_query($query)){
      $this->errors['sql'][] = "Failed to execute multi query! INFO: ({$this->mysqli->errno}) \"{$this->mysqli->error}\". QUERY: {$query}";
      $return = FALSE;
    } else {
      $return = array();  
      reset($this->multi_query);
        do {
            if ($result = $this->mysqli->store_result()) {
          $current = key($this->multi_query);
          if(!empty($this->multi_query[$current]['mq_id'])){
            $return[$this->multi_query[$current]['mq_id']] = array();
          } else {
            $return[] = array();
          }
          $key = key($return);
          switch($this->multi_query[$current]['type']){
            case 'select':
              $return[$key] = array();
              $this->select_parse_result($result, $return, $this->multi_query[$current]['conf']);
              break;
          }
            }
        } while ($this->mysqli->next_result());
    }
    $successful = !($return === false);
    $this->log_query($query, $this->mysqli->errno, $this->mysqli->error, $successful, $return);
    return $return;
  }
}
?>